#include "RankingSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include "UserProfileSony.h"
#include "SignInSony.h"
#if NP_HAS_FRIENDS_LIST
 #include "FriendsListSony.h"
#endif
using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedRanking gRanking;

	DO_EXPORT( bool, PrxRankingGetLastError) (ResultCode* result)
	{
		return gRanking.GetLastError(result);
	}
	
	DO_EXPORT( ErrorCode, PrxRankingRegisterCache ) (int boardLineCount, int writeLineCount, bool friendCache, int rangeLineCount)
	{
		return gRanking.RegisterCache(boardLineCount, writeLineCount, friendCache, rangeLineCount);
	}

	DO_EXPORT( bool, PrxRankingRegisterScoreIsBusy ) ()
	{
		return gRanking.RegisterScoreIsBusy();
	}

	DO_EXPORT( ErrorCode, PrxRankingRegisterScore ) (int boardID, Int64 score, const char* comment)
	{
		return gRanking.RegisterScore(boardID, score, comment, NULL ,0);
	}

	DO_EXPORT( ErrorCode, PrxRankingRegisterScoreWithData ) (int boardID, Int64 score, const char* comment, const char* data, int datalen)
	{
		return gRanking.RegisterScore(boardID, score, comment, data, datalen);
	}

	DO_EXPORT( bool, PrxRankingRefreshOwnRankIsBusy ) ()
	{
		return gRanking.RefreshOwnRankIsBusy();
	}

	DO_EXPORT( ErrorCode, PrxRankingRefreshOwnRank ) (int boardID)
	{
		return gRanking.RefreshOwnRank(boardID);
	}

	DO_EXPORT( ErrorCode, PrxRankingGetOwnRank ) (Rank* rank)
	{
		return gRanking.GetOwnRank(rank);
	}

	DO_EXPORT( bool, PrxRankingRefreshFriendRankIsBusy ) ()
	{
		return gRanking.RefreshFriendRankIsBusy();
	}

	DO_EXPORT( ErrorCode, PrxRankingRefreshFriendRank ) (int boardID)
	{
		return gRanking.RefreshFriendRank(boardID);
	}

	DO_EXPORT( void, PrxRankingLockFriendRankList ) ()
	{
		gRanking.LockFriendsRankList();
	}

	DO_EXPORT( void, PrxRankingUnlockFriendRankList ) ()
	{
		gRanking.UnlockFreindsRankList();
	}

	DO_EXPORT( int, PrxRankingGetFriendRankCount ) ()
	{
		return (int)gRanking.GetFriendsRankCount();
	}

	DO_EXPORT( ErrorCode, PrxRankingGetFriendRank ) (int index, Rank* rank)
	{
		return gRanking.GetFriendRank(index, rank);
	}

	DO_EXPORT( bool, PrxRankingRefreshRankListIsBusy ) ()
	{
		return gRanking.RefreshRankListIsBusy();
	}

	DO_EXPORT( ErrorCode, PrxRankingRefreshRankList ) (int boardID, int firstIndex, int count)
	{
		return gRanking.RefreshRankList(boardID, firstIndex, count);
	}

	DO_EXPORT( void, PrxRankingLockRankList ) ()
	{
		gRanking.LockRankList();
	}

	DO_EXPORT( void, PrxRankingUnlockRankList ) ()
	{
		gRanking.UnlockRankList();
	}

	DO_EXPORT( int, PrxRankingGetRankListCount ) ()
	{
		return (int)gRanking.GetRankListCount();
	}

	DO_EXPORT( ErrorCode, PrxRankingGetRank ) (int index, Rank* rank)
	{
		return gRanking.GetRank(index, rank);
	}

	DO_EXPORT( int, PrxRankingGetTotalRankCount ) ()
	{
		return gRanking.GetTotalRankCount();
	}

	CachedRanking::CachedRanking()
		: m_LastResult("Ranking")
		, m_RefreshRangeScoreBusy(false)
		, m_RefreshFriendRankBusy(false)
		, m_RegisterScoreBusy(false)
		, m_RefreshOwnRankBusy(false)
		, m_TotalScoreCount(0)
	{
	}

	bool CachedRanking::RegisterScoreIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RegisterScoreBusy || m_RefreshOwnRankBusy;
	}

	bool CachedRanking::RefreshOwnRankIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshOwnRankBusy || m_RegisterScoreBusy;
	}

	bool CachedRanking::RefreshFriendRankIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshFriendRankBusy;
	}

	void CachedRanking::LockFriendsRankList()
	{
		m_Lock.Lock();
	}

	void CachedRanking::UnlockFreindsRankList()
	{
		m_Lock.Unlock();
	}

	void CachedRanking::ClearFriendRankList()
	{
		for(unsigned int i=0; i<m_FriendRanks.size(); i++)
		{
			delete m_FriendRanks[i];
		}
		m_FriendRanks.clear();
	}

	int CachedRanking::GetFriendsRankCount()
	{
		return static_cast<int>(m_FriendRanks.size());
	}

	bool CachedRanking::RefreshRankListIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshRangeScoreBusy;
	}

	void CachedRanking::LockRankList()
	{
		m_Lock.Lock();
	}

	void CachedRanking::UnlockRankList()
	{
		m_Lock.Unlock();
	}

	void CachedRanking::ClearRankList()
	{
		for(unsigned int i=0; i<m_Ranks.size(); i++)
		{
			delete m_Ranks[i];
		}
		m_Ranks.clear();
	}

	int CachedRanking::GetRankListCount()
	{
		return static_cast<int>(m_Ranks.size());
	}

	int CachedRanking::GetTotalRankCount()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_TotalScoreCount;
	}

	bool CachedRanking::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;
		int slot = 0;

		switch(event.event)
		{
		case Event::rankingScoreRegistered:				// An event generated when a ranking score has been registered.
			// Update the cached user rank with the new rank.
			m_NewRank.rank = m_FutureTempRank.get()->tempRank;
			m_NewRank.highestRank = m_NewRank.rank;
			m_NewRank.serialRank = m_NewRank.rank;

			if (slot >= 0)
			{
				CachedRank *ownRank = &m_OwnRank[slot];
				*ownRank = m_NewRank;
			}

			m_RegisterScoreBusy = false;
			handled = true;
			Messages::AddMessage(Messages::kNPToolKit_RankingNewBestScore);
			break;

		case Event::rankingScoreRegisteredFailNotBest:	// An event generated when a score being registered is not the best score.
			m_RegisterScoreBusy = false;
			handled = true;
			Messages::AddMessage(Messages::kNPToolKit_RankingNotBestScore);
			break;

		case Event::rankingScoreRegisteredFail:			// An event generated when an attempt to register a ranking score has failed.
			m_RegisterScoreBusy = false;
			handled = true;
			if( (m_FutureTempRank.getError() == SCE_TOOLKIT_NP_RANKING_NOT_HIGH_SCORE) ||
				(m_FutureTempRank.getError() == SCE_NP_COMMUNITY_SERVER_ERROR_NOT_BEST_SCORE) )
			{
				Messages::AddMessage(Messages::kNPToolKit_RankingNotBestScore);
			}
			else
			{
				m_LastResult.SetResultSCE(m_FutureTempRank.getError(), true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_RankingError);
			}
			break;

		case Event::rankingUserRankRetrieved:			// An event generated when a user's rank has been retrieved.
			if(event.returnCode < 0)
			{
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_RankingError);
				m_RefreshOwnRankBusy = false;
				handled = true;
				break;
			}

			if(m_FutureOwnRank.get()->rankData.hasData)
			{
				// If the currently cached rank is worse than the server rank use the
				// server rank, otherwise the server rank must have not updated yet so
				// use the cached rank (as long as it's valid, i.e. rank != 0).
				UserRankInformation* rankInfo = m_FutureOwnRank.get();

				int slot = 0;
				CachedRank *ownRank = &m_OwnRank[slot];	// get the current cached version of the players data
				if((slot >= 0) && ((ownRank->serialRank == 0) || (rankInfo->rankData.rankData.serialRank < ownRank->serialRank) || (ownRank->boardId != m_RefreshOwnBoardID)) )
				{
					ownRank->NpId = rankInfo->rankData.rankData.npId;
					ownRank->PcId = rankInfo->rankData.rankData.pcId;
					ownRank->serialRank = rankInfo->rankData.rankData.serialRank;
					ownRank->rank = rankInfo->rankData.rankData.rank;
					ownRank->highestRank = rankInfo->rankData.rankData.highestRank;
					ownRank->hasGameData = rankInfo->rankData.rankData.hasGameData;
					ownRank->score = rankInfo->rankData.rankData.scoreValue;

					ownRank->comment = rankInfo->comment.data;
					ownRank->gameInfoSize = 0;
				//	if (ownRank->hasGameData)
					{
						ownRank->gameInfoSize = SCE_NP_SCORE_GAMEINFO_SIZE;
						memcpy(ownRank->gameInfoData, rankInfo->gameInfo.nativeData, ownRank->gameInfoSize);
					}
					ownRank->boardId = rankInfo->boardId;
					ownRank->provisional = false;
				}
			}

			m_RefreshOwnRankBusy = false;
			handled = true;
			Messages::AddMessage(Messages::kNPToolKit_RankingGotOwnRank);
			break;

		case Event::rankingUserRankRetrievedFail:		// An event generated when a user's rank failed to be retrieved.
			slot = 0;
			if (slot>=0)
			{
				CachedRank *ownRank = &m_OwnRank[slot];
				if(m_FutureOwnRank.getError() == (int)SCE_NP_COMMUNITY_SERVER_ERROR_GAME_RANKING_NOT_FOUND)
				{
					if((ownRank->rank <= 0) || (ownRank->boardId != m_RefreshOwnBoardID))
					{
						// Unranked.
						ownRank->clear();
						ownRank->NpId = gUserProfile.GetNpID();
						ownRank->boardId = m_RefreshOwnBoardID;
					}
					Messages::AddMessage(Messages::kNPToolKit_RankingGotOwnRank);
				}
				else
				{
					ownRank->provisional = true;
					m_LastResult.SetResultSCE(m_FutureOwnRank.getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_RankingError);
				}
			}


			m_RefreshOwnRankBusy = false;
			handled = true;
			break;

		case Event::rankingFriendsRetrieved:			// An event generated when the ranks and scores belonging a user's friend have been retrieved.
			{
				ClearFriendRankList();

				FriendsRankInformation* rankInfo = m_FutureFriendRank.get();
				for(int i=0; i<m_FutureFriendRank.get()->numFriends; i++)
				{
					CachedRank* rank = new CachedRank();

					rank->NpId = rankInfo->rankData[i].rankData.npId;
					rank->PcId = rankInfo->rankData[i].rankData.pcId;
					rank->serialRank = rankInfo->rankData[i].rankData.serialRank;
					rank->rank = rankInfo->rankData[i].rankData.rank;
					rank->highestRank = rankInfo->rankData[i].rankData.highestRank;
					rank->hasGameData = rankInfo->rankData[i].rankData.hasGameData;
					rank->score = rankInfo->rankData[i].rankData.scoreValue;
					rank->gameInfoSize = 0;

					rank->comment = rankInfo->comment[i].data;
				//	if (rank->hasGameData)
					{
						rank->gameInfoSize = SCE_NP_SCORE_GAMEINFO_SIZE;
						memcpy(rank->gameInfoData, rankInfo->gameInfo[i].nativeData, rank->gameInfoSize);
					}
					rank->boardId = rankInfo->boardId;
					rank->provisional = false;

					m_FriendRanks.push_back(rank);
				}
			}

			m_RefreshFriendRankBusy = false;
			handled = true;
			Messages::AddMessage(Messages::kNPToolKit_RankingGotFriendRank);
			break;

		case Event::rankingFriendsRetrievedFail:		// An event generated when the rank of a user's friend could not be retrieved.
			if( (m_FutureFriendRank.getError() == (int)SCE_NP_COMMUNITY_SERVER_ERROR_GAME_RANKING_NOT_FOUND) ||
				(m_FutureFriendRank.getError() == (int)SCE_TOOLKIT_NP_RANKING_FRIEND_LIST_EMPTY) )
			{
				// Ranking board is empty so just clear the list.
				ClearRankList();
				Messages::AddMessage(Messages::kNPToolKit_RankingGotFriendRank);
			}
			else
			{
				m_LastResult.SetResultSCE(m_FutureFriendRank.getError(), true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_RankingError);
			}
			m_RefreshFriendRankBusy = false;
			handled = true;
			break;

		case Event::rankingFriendsRetrievedFailNoFriends:	// An event generated when a user has no friends so ranking service cannot perform a request.
			// No friends so just clear the list.
			ClearRankList();
			Messages::AddMessage(Messages::kNPToolKit_RankingGotFriendRank);
			m_RefreshFriendRankBusy = false;
			handled = true;
			break;

		case Event::rankingRangeRetrieved:				// An event generated when a range of ranks from a scoreboard has been retrieved.
			{
				ClearRankList();

				RankInformation* rankInfo = m_FutureRangeScore.get();
				m_TotalScoreCount = rankInfo->totalRecord;

				for(int i=0; i<rankInfo->rankReturned; i++)
				{
					CachedRank* rank = new CachedRank();

					rank->NpId = rankInfo->rankData[i].npId;
					rank->PcId = rankInfo->rankData[i].pcId;
					rank->serialRank = rankInfo->rankData[i].serialRank;
					rank->rank = rankInfo->rankData[i].rank;
					rank->highestRank = rankInfo->rankData[i].highestRank;
					rank->hasGameData = rankInfo->rankData[i].hasGameData;
					rank->score = rankInfo->rankData[i].scoreValue;
					rank->gameInfoSize =0;
					rank->comment = rankInfo->comment[i].data;
			//		if (rank->hasGameData)
					{
						rank->gameInfoSize = SCE_NP_SCORE_GAMEINFO_SIZE;
						memcpy(rank->gameInfoData, rankInfo->gameInfo[i].nativeData, rank->gameInfoSize);
					}
					rank->boardId = rankInfo->boardId;
					rank->provisional = false;

					m_Ranks.push_back(rank);
				}
			}

			m_RefreshRangeScoreBusy = false;
			handled = true;
			Messages::AddMessage(Messages::kNPToolKit_RankingGotRankList);
			break;

		case Event::rankingRangeRetrievedFail:			// An event generated when a range of ranks from a scoreboard failed to be retrieved.
			if(m_FutureRangeScore.getError() == (int)SCE_NP_COMMUNITY_SERVER_ERROR_GAME_RANKING_NOT_FOUND)
			{
				// Ranking board is empty so just clear the list.
				ClearRankList();
				Messages::AddMessage(Messages::kNPToolKit_RankingGotRankList);
			}
			else
			{
				m_LastResult.SetResultSCE(m_FutureRangeScore.getError(), true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_RankingError);
			}
			m_TotalScoreCount = 0;
			m_RefreshRangeScoreBusy = false;
			handled = true;
			break;

		case Event::rankingMatchingBoardFound:			// An event generated when a board matching the arguments was found.
			Messages::LogWarning("Ranking event not handled: event=%d\n", event.event);
			handled = true;
			break;

		case Event::rankingHighScore:					// An event generated when a higher score was already recorded in the cache.
		case Event::rankingServerError:					// An event generated when an error occurred with the ranking server.
		case Event::rankingCommunityError:				// An event generated when an error occurred communicating with the ranking server.
		case Event::rankingMemoryError:					// An event generated when the ranking service cannot allocate anymore memory for the cache.
			Messages::LogWarning("Ranking event not handled: event=%d\n", event.event);
			handled = true;
			break;

		default:
			Messages::LogWarning("Unexpected event from ranking service: event=%d\n", event.event);
			handled = true;
			break;
		}

		return handled;
	}

	ErrorCode CachedRanking::RegisterCache(int boardLineCount, int writeLineCount, bool friendCache, int rangeLineCount)
	{
		m_LastResult.Reset();
		int ret = Ranking::Interface::registerCache(boardLineCount, writeLineCount, rangeLineCount, friendCache);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		Messages::AddMessage(Messages::kNPToolKit_RankingCacheRegistered);
		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::RegisterScore(int boardID, Int64 score, const char* comment, const char *attacheddata, int attacheddatalength)
	{
		if(RegisterScoreIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true, __FUNCTION__, __LINE__);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		sce::Toolkit::NP::RegisterScore registerScore;
		memset(&registerScore,0, sizeof(registerScore));

		registerScore.boardId = boardID;
		registerScore.score = score;
		int len = strlen(comment);
		if(len > SCE_NP_SCORE_COMMENT_MAXLEN)
		{
			len = SCE_NP_SCORE_COMMENT_MAXLEN;
		}
		memcpy(&registerScore.comment, comment, len+1);

		if(attacheddata)
		{
			if (attacheddatalength > SCE_NP_SCORE_GAMEINFO_SIZE)
			{
				attacheddatalength = SCE_NP_SCORE_GAMEINFO_SIZE;
			}
			memcpy(&registerScore.gameInfo.nativeData, attacheddata, attacheddatalength);
		}
		// Retrieve tempRank back;
		m_FutureTempRank.reset();

		m_NewRank.clear();
		m_NewRank.NpId = gUserProfile.GetNpID();
		m_NewRank.boardId = boardID;
		m_NewRank.comment = comment;
		m_NewRank.score = score;
		m_NewRank.provisional = true;

		m_NewRank.gameInfoSize = attacheddatalength;
		memcpy(m_NewRank.gameInfoData, attacheddata, attacheddatalength);

		int ret = Ranking::Interface::registerScore(&m_FutureTempRank, &registerScore, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_RegisterScoreBusy = true;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::RefreshOwnRank(int boardID)
	{
		if(RefreshOwnRankIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true, __FUNCTION__, __LINE__);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		m_RefreshOwnBoardID = (SceNpScoreBoardId)boardID;

		SceNpId tempNpId;
		memset(&tempNpId, 0, sizeof(tempNpId)); // Pass in an empty SceNpId, this will retrieve the current user's rank by default.

		int ret = Ranking::Interface::displayUserRank(&m_FutureOwnRank, tempNpId, boardID, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_RefreshOwnRankBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::RefreshFriendRank(int boardID)
	{
		//	This is required on PS3 as otherwise when the user has no friends, the Ranking::Interface::displayFriendRank fails and leaves the
		//	m_FutureFriendRank.m_errorCode permanently set at _SCE_FUTURE_IMPL_BUSY. It is then not possible to clear it and subsequent calls to this
		//	function fail internall when messages are unable to be sent. Guard against the user doing this but the check here
		if(UnityPlugin::gFriendsList.GetFriendCount()==0)
		{
			DBG_LOG("Calling friend rank when user has no friends");
			return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
		}

		if(RefreshFriendRankIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true, __FUNCTION__, __LINE__);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		m_FutureFriendRank.reset();
				
		int ret = Ranking::Interface::displayFriendRank(&m_FutureFriendRank, boardID, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		
		m_RefreshFriendRankBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::RefreshRankList(int boardID, int firstIndex, int count)
	{
		if(RefreshRankListIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true, __FUNCTION__, __LINE__);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_FutureRangeScore.reset();

		m_LastResult.Reset();

		if(firstIndex < 1)
		{
			firstIndex = 1;
			Messages::LogWarning("Ranking::RefreshRankList() firstIndex must be >= 1\n");
		}

		if(count > 30)
		{
			count = 30;
			Messages::LogWarning("Ranking::RefreshRankList() count must be <= 30\n");
		}

		int ret = Ranking::Interface::displayRangeOfRanks(&m_FutureRangeScore, boardID, firstIndex, count, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_RefreshRangeScoreBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::GetRank(unsigned int index, Rank* rank)
	{
		if(index >= m_Ranks.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedRank* cached = m_Ranks[index];
		rank->onlineID = cached->NpId.handle.data;
		rank->PcId = cached->PcId;
		rank->serialRank = cached->serialRank;
		rank->rank = cached->rank;
		rank->highestRank = cached->highestRank;
		rank->hasGameData = cached->hasGameData;
		rank->score = cached->score;
		rank->gameInfoSize = cached->gameInfoSize;
		rank->comment = cached->comment.c_str();

		rank->gameInfoData = cached->gameInfoData;	// ptr assigned from allocated data 
		rank->boardId = cached->boardId;
		rank->provisional = cached->provisional;

		return m_LastResult.GetResult();
	}
	
	ErrorCode CachedRanking::GetOwnRank(Rank* rank)
	{
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int slot = 0;
		if (slot >= 0)
		{
			CachedRank *ownRank = &m_OwnRank[slot];

			rank->onlineID = ownRank->NpId.handle.data;
			rank->PcId = ownRank->PcId;
			rank->serialRank = ownRank->serialRank;
			rank->rank = ownRank->rank;
			rank->highestRank = ownRank->highestRank;
			rank->hasGameData = ownRank->hasGameData;
			rank->score = ownRank->score;
			rank->gameInfoSize = ownRank->gameInfoSize;
			rank->comment = ownRank->comment.c_str();

			rank->gameInfoData = ownRank->gameInfoData;	// ptr assigned from allocated data 
			rank->boardId = ownRank->boardId;
			rank->provisional = ownRank->provisional;
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedRanking::GetFriendRank(unsigned int index, Rank* rank)
	{
		if(index >= m_FriendRanks.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedRank* cached = m_FriendRanks[index];
		rank->onlineID = cached->NpId.handle.data;
		rank->PcId = cached->PcId;
		rank->serialRank = cached->serialRank;
		rank->rank = cached->rank;
		rank->highestRank = cached->highestRank;
		rank->hasGameData = cached->hasGameData;
		rank->score = cached->score;
		rank->gameInfoSize = cached->gameInfoSize;
		rank->comment = cached->comment.c_str();

		rank->gameInfoData = cached->gameInfoData;	// ptr assigned from allocated data 
		rank->boardId = cached->boardId;
		rank->provisional = cached->provisional;

		return m_LastResult.GetResult();
	}

}
